<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

trait WPMR_Account_Mgt {


	function is_registered() {
		return $this->get_setting( 'user' );
	}

	function wpmr_cli_register( $email, $fn, $ln, $echo = false ) {
		global $wp_version;
		$args     = array(
			'user' => array(
				'fn'    => $fn,
				'ln'    => $ln,
				'email' => $email,
				'key'   => site_url(),
			),
			'diag' => array(
				'site_url'       => trailingslashit( site_url() ),
				'php'            => phpversion(),
				'web_server'     => empty( $_SERVER['SERVER_SOFTWARE'] ) ? 'none' : sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ),
				'wp'             => $wp_version,
				'plugin_version' => $this->plugin_data['Version'],
				'cachebust'      => microtime( 1 ),
			),
		);
		$args     = $this->encode( $args );
		$url      = add_query_arg( 'wpmr_action', 'wpmr_register', add_query_arg( 'reg_details', $args, WPMR_SERVER ) );
		$response = wp_safe_remote_request(
			$url,
			array(
				'blocking' => true,
				'timeout'  => $this->timeout,
			)
		);
		if ( is_wp_error( $response ) ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( $response->get_error_message() );
				} else {
					echo esc_html( $response->get_error_message() );
				}
			} else {
				return;
			}
		}
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 != $status_code ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( 'Error: Status_code ' . $status_code );
				} else {
					echo 'Error: Status_code ' . esc_html( $status_code );
				}
			} else {
				return;
			}
		}
		$response = wp_remote_retrieve_body( $response );
		if ( empty( $response ) || is_null( $response ) ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( 'No response. Registration Failed.' );
				} else {
					echo 'No response. Registration Failed.';
				}
			} else {
				return;
			}
		}
		$data = json_decode( $response, true );
		if ( ! $data ) {
			WP_CLI::error( 'Invalid Server Response. Registration Failed.' );
		}
		if ( ! isset( $data['error'] ) ) {
			$this->update_setting( 'user', $data );
		}
		if ( $echo ) {
			if ( $this->wpmr_iscli() ) {
				WP_CLI::success( 'Registration complete. Please use ' . WP_CLI::colorize( '%Y' . $data['user_email'] . '%n' ) . ' as your USER ID.' );
			} else {
				echo 'Registration complete. Please use <strong>' . esc_html( $data['user_email'] ) . '</strong> as your USER ID.';
			}
		} else {
			return true;
		}
	}

	function wpmr_web_register() {
		check_ajax_referer( 'wpmr_web_register', 'wpmr_web_register_nonce' );
		if ( ! current_user_can( $this->cap ) ) {
			return;
		}
		$this->flog( 'Starting web registration' );
		$start_time = microtime( true );
		global $wp_version;

		$user_data = isset( $_REQUEST['user'] ) ? wp_unslash( $_REQUEST['user'] ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array is sanitized field by field below

		if ( empty( $user_data ) || empty( $user_data['email'] ) || empty( $user_data['fn'] ) || empty( $user_data['ln'] ) ) {
			wp_send_json( array( 'error' => 'Please fill all details' ) );
		}
		if ( ! filter_var( sanitize_email( $user_data['email'] ), FILTER_VALIDATE_EMAIL ) ) {
			wp_send_json( array( 'error' => 'Invalid email' ) );
		}

		// Sanitize user data
		$sanitized_user = array(
			'email' => sanitize_email( $user_data['email'] ),
			'fn'    => sanitize_text_field( $user_data['fn'] ),
			'ln'    => sanitize_text_field( $user_data['ln'] ),
		);

		$args     = array(
			'user' => $sanitized_user,
			'diag' => array(
				'site_url'       => trailingslashit( site_url() ),
				'php'            => phpversion(),
				'web_server'     => empty( $_SERVER['SERVER_SOFTWARE'] ) ? 'none' : sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ),
				'wp'             => $wp_version,
				'plugin_version' => $this->plugin_data['Version'],
				'cachebust'      => microtime( 1 ),
			),
		);
		$args     = $this->encode( $args );
		$url      = add_query_arg( 'wpmr_action', 'wpmr_register', add_query_arg( 'reg_details', $args, WPMR_SERVER ) );
		$response = wp_safe_remote_request(
			$url,
			array(
				'blocking' => true,
				'timeout'  => $this->timeout,
			)
		);
		$this->flog( 'Web registration completed in ' . ( microtime( true ) - $start_time ) . ' seconds' );
		$start_time = microtime( true );
		if ( is_wp_error( $response ) ) {
			wp_send_json( array( 'error' => $response->get_error_message() ) );
		}
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 != $status_code ) {
			wp_send_json( array( 'error' => $status_code ) );
		}
		$response = wp_remote_retrieve_body( $response );
		if ( empty( $response ) || is_null( $response ) ) {
			wp_send_json( array( 'error' => 'No response. Registration Failed.' ) );
		}
		$data = json_decode( $response, true );
		if ( ! $data ) {
			wp_send_json( array( 'error' => 'Invalid response from server.' ) );
		}
		if ( ! isset( $data['error'] ) ) {
			$this->update_setting( 'user', $data );
			$this->refresh_checksums_async(); // trigger in background so user doesn't have to wait
			$this->flog( 'Checksums fetched in ' . ( microtime( true ) - $start_time ) . ' seconds' );
			$start_time = microtime( true );
			$this->update_definitions_cli( false );
			$this->flog( 'Definitions updated in ' . ( microtime( true ) - $start_time ) . ' seconds' );
		}
		wp_send_json( $data );
	}

	function clear_license_status() {
		delete_transient( 'WPMR_license_status' );
		return true;
	}

	function deactivate_license() {
		$key      = $this->get_setting( 'license_key' );
		$response = $key ? $this->get_license_api_response( 'deactivate_license', $key ) : new WP_Error( 'error', 'No license key found for ' . __FUNCTION__ );
		// fail silently as we are only called during plugin deactivation;
		$this->delete_setting( 'license_key' );
		$this->clear_license_status();
	}

	function format_license_status_data( $from_cache = 0 ) {
		if ( ! $this->get_setting( 'license_key' ) ) {
			return json_encode( array( 'error' => 'No license key found to format.' ) );
		}

		$status = get_transient( 'WPMR_license_status' );

		if ( ! $from_cache || ! $status ) {
			$status = $this->get_license_api_response( 'check_license', $this->get_setting( 'license_key' ) );
		}

		if ( ! $status ) {
			return json_encode( array( 'error' => 'Unable to retrieve license status' ) );
		}

		// Build the response data
		$response_data = array(
			'success'  => true,
			'license'  => isset( $status['license'] ) ? $status['license'] : 'unknown',
			'raw_data' => $status,
		);

		// Add formatted status information
		if ( isset( $status['license'] ) ) {
			$response_data['license_status'] = ucwords( preg_replace( '/[^A-Za-z0-9 ]/', ' ', $status['license'] ) );
		}

		// Add error/reason if present
		if ( isset( $status['error'] ) ) {
			$response_data['error_reason'] = ucwords( preg_replace( '/[^A-Za-z0-9 ]/', ' ', $status['error'] ) );
		}

		// Add customer information
		if ( ! empty( $status['customer_email'] ) ) {
			$response_data['customer_email'] = $status['customer_email'];
		}

		if ( ! empty( $status['customer_name'] ) ) {
			$response_data['customer_name'] = $status['customer_name'];
		}

		// Add activation information
		if ( ! empty( $status['site_count'] ) && isset( $status['activations_left'] ) ) {
			$limit = 'Unlimited';
			if ( isset( $status['license_limit'] ) && $status['license_limit'] !== null ) {
				$limit = $status['license_limit'] == 0 ? 'Unlimited' : $status['license_limit'];
			}
			$response_data['activations'] = array(
				'used'      => $status['site_count'],
				'limit'     => $limit,
				'remaining' => $status['activations_left'],
			);
		}

		// Add expiration information
		if ( ! empty( $status['expires'] ) ) {
			if ( $status['expires'] == 'lifetime' ) {
				$response_data['expires']           = 'Never';
				$response_data['expires_formatted'] = 'Lifetime';
			} else {
				$response_data['expires'] = $status['expires'];
				try {
					$timezone_string = function_exists( 'wp_timezone_string' ) ? wp_timezone_string() : $this->timezone_string_compat();
					$timezone        = new DateTimeZone( $timezone_string );
					$date            = new DateTime( $status['expires'], new DateTimeZone( 'UTC' ) );
					$date->setTimezone( $timezone );
					$response_data['expires_formatted'] = $date->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) . ' ' . $timezone_string;
				} catch ( Exception $e ) {
					$response_data['expires_formatted'] = $status['expires'];
				}
			}
		}

		// Add renewal link if expired
		if ( isset( $status['license'] ) && $status['license'] == 'expired' ) {
			$response_data['renewal_link'] = 'https://malcure.com/?p=168&edd_license_key=' . $this->get_setting( 'license_key' ) . '&edd_action=apply_license_renewal&utm_source=pluginexpired&utm_medium=web&utm_campaign=wpmr';
		}

		return json_encode( $response_data );
	}

	function get_license_api_response( $action, $license_key = '', $silent = false ) {
		if ( empty( $license_key ) ) {
			$license_key = $this->get_setting( 'license_key' );
		}
		if ( empty( $license_key ) ) {
			$this->flog( 'License key not found.' );
			return null;
		}
		$url      = MALCURE_API . '?edd_action=' . $action . '&item_id=1725&license=' . urlencode( $license_key ) . '&url=' . site_url() . '&cachebust=' . microtime( true );
		$response = wp_safe_remote_request( $url, array( 'timeout' => $this->timeout ) );

		$this->flog( 'License API Request: ' . $url );

		if ( is_wp_error( $response ) ) {
			$this->flog( 'API Error: ' . $response->get_error_message() );
			return null;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $status_code ) {
			$this->flog( 'HTTP Error: ' . $status_code );
			return null;
		}

		$body = wp_remote_retrieve_body( $response );
		$this->flog( 'License API Response Body: ' );
		$this->flog( $body );
		$data = json_decode( $body, true );

		if ( is_null( $data ) ) {
			$this->flog( 'Unparsable response: ' . $body );
			return null;
		}
		return $data;
	}

	function is_advanced_edition() {
		$status = get_transient( 'WPMR_license_status' );
		if ( ! $status ) {
			$key = $this->get_setting( 'license_key' );
			if ( empty( $key ) ) {
				return false;
			}

			$status = $this->get_license_api_response( 'check_license', $key );
			if ( is_wp_error( $status ) || empty( $status['success'] ) ) {
				return false;
			}

			$this->save_license_status( $status );
		}

		return $status['license'] === 'valid';
	}

	function is_advanced_edition_expired() {
		$this->is_advanced_edition(); // Ensure the transient is refreshed if necessary
		$status = get_transient( 'WPMR_license_status' );
		return ( ! empty( $status['license'] ) && $status['license'] === 'expired' );
	}

	function save_license_status( $status ) {
		// $this->flog( $status );
		set_transient( 'WPMR_license_status', $status, 24 * HOUR_IN_SECONDS );
		return $status;
	}

	function save_gsc_profile() {
		if ( isset( $_REQUEST['wpmr-action'] ) && $_REQUEST['wpmr-action'] == 'oauth' ) {
			$origin_nonce = isset( $_REQUEST['origin_nonce'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['origin_nonce'] ) ) : '';
			if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['origin_nonce'] ) ), 'wpmr_gscapi' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Validated via wp_verify_nonce
				return;
			}
			$success = isset( $_REQUEST['success'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['success'] ) ) : '';
			if ( current_user_can( $this->cap ) && $success ) {
				$wpmr_reg = $this->get_setting( 'user' );
				if ( $wpmr_reg ) {
					$wpmr_reg['gsc'] = true;
					$this->update_setting( 'user', $wpmr_reg );
				}
			} else {
			}
			wp_redirect( esc_url( get_admin_url( null, 'options-general.php?page=wpmr' ) ), 302 );
			exit;
		}
		if ( isset( $_REQUEST['wpmr-action'] ) && $_REQUEST['wpmr-action'] == 'revoke' && isset( $_REQUEST['success'] ) && $_REQUEST['success'] == '1' ) {
			$wpmr_reg = $this->get_setting( 'user' );
			if ( $wpmr_reg && ! empty( $wpmr_reg['gsc'] ) ) {
				unset( $wpmr_reg['gsc'] );
				$this->update_setting( 'user', $wpmr_reg );
			}
		}
	}

	function ajax_get_license_status() {
		check_ajax_referer( 'wpmr_fetch_license', 'wpmr_fetch_license_nonce' );
		if ( ! current_user_can( $this->cap ) ) {
			wp_send_json_error( 'Insufficient permissions.' );
		}

		$license_status_json = $this->format_license_status_data();
		$license_data        = json_decode( $license_status_json, true );

		if ( $license_data && isset( $license_data['success'] ) && $license_data['success'] ) {
			wp_send_json_success( $license_data );
		} else {
			$error_message = isset( $license_data['error'] ) ? $license_data['error'] : 'Failed to retrieve license status';
			wp_send_json_error( $error_message );
		}
	}
}
